<?php
declare(strict_types=1);

$cfgFile = __DIR__ . '/../config.php';
if (file_exists($cfgFile)) die('Installer ist gesperrt (config.php existiert).');

$appName = trim((string)($_POST['app_name'] ?? 'Bibliothek'));
$adminName = trim((string)($_POST['admin_name'] ?? 'Admin'));
$pin1 = (string)($_POST['admin_pin'] ?? '');
$pin2 = (string)($_POST['admin_pin2'] ?? '');

if ($appName === '' || $adminName === '' || $pin1 === '' || $pin1 !== $pin2) die('Ungültige Eingaben (Name/PIN).');
if (!preg_match('/^[0-9]{3,12}$/', $pin1)) die('PIN muss 3–12 Ziffern haben.');
if (!extension_loaded('pdo_sqlite')) die('PDO SQLite fehlt.');

$dataDir = __DIR__ . '/../data';
$uploadsDir = __DIR__ . '/../uploads/books';
if (!is_dir($dataDir) || !is_writable($dataDir)) die('/data ist nicht beschreibbar.');
if (!is_dir($uploadsDir) || !is_writable($uploadsDir)) die('/uploads/books ist nicht beschreibbar.');

$dbPath = realpath($dataDir) . DIRECTORY_SEPARATOR . 'library.db';

try {
  $pdo = new PDO('sqlite:' . $dbPath, null, null, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  $pdo->exec("PRAGMA foreign_keys = ON;");

  $schema = "
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      pin_hash TEXT,
      role TEXT NOT NULL DEFAULT 'member',
      created_at TEXT NOT NULL DEFAULT (datetime('now'))
    );

    CREATE TABLE IF NOT EXISTS books (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      title TEXT NOT NULL,
      author TEXT,
      isbn TEXT,
      note TEXT,
      image TEXT,
      created_at TEXT NOT NULL DEFAULT (datetime('now'))
    );

    CREATE TABLE IF NOT EXISTS loans (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      book_id INTEGER NOT NULL,
      user_id INTEGER NOT NULL,
      loaned_at TEXT NOT NULL DEFAULT (datetime('now')),
      returned_at TEXT,
      FOREIGN KEY(book_id) REFERENCES books(id) ON DELETE CASCADE,
      FOREIGN KEY(user_id) REFERENCES users(id) ON DELETE CASCADE
    );

    CREATE UNIQUE INDEX IF NOT EXISTS idx_loans_book_active
    ON loans(book_id)
    WHERE returned_at IS NULL;
  ";
  $pdo->exec($schema);

  $pinHash = password_hash($pin1, PASSWORD_DEFAULT);
  $stmt = $pdo->prepare("INSERT INTO users (name, pin_hash, role) VALUES (?, ?, 'admin')");
  $stmt->execute([$adminName, $pinHash]);

  $cfg = "<?php\nreturn [\n"
       . "  'installed' => true,\n"
       . "  'app_name' => " . var_export($appName, true) . ",\n"
       . "  'db_path' => __DIR__ . '/data/library.db',\n"
       . "];\n";
  file_put_contents($cfgFile, $cfg);

} catch (Throwable $e) {
  die('Installation fehlgeschlagen: ' . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8'));
}

$base = rtrim(dirname(dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/');
$adminUrl = $base . '/admin/';
$frontUrl = $base . '/mobirise/ausleihe.php';
?>
<!doctype html>
<html lang="de">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Installation abgeschlossen</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<div class="container py-5" style="max-width: 860px;">
  <div class="card shadow-sm">
    <div class="card-body">
      <h1 class="h4 mb-3">✅ Installation abgeschlossen</h1>
      <div class="alert alert-warning">
        Bitte den Ordner <code>/install</code> jetzt löschen oder sperren.
      </div>

      <div class="d-flex flex-wrap gap-2">
        <a class="btn btn-primary" href="<?= htmlspecialchars($adminUrl) ?>">Zum Adminbereich</a>
        <a class="btn btn-outline-primary" href="<?= htmlspecialchars($frontUrl) ?>">Zur Ausleihe (Frontend)</a>
      </div>

      <hr>
      <div class="text-muted small">
        Admin-Login: <code>/admin/</code> (Name + PIN)<br>
        Frontend: <code>/mobirise/ausleihe.php</code>
      </div>
    </div>
  </div>
</div>
</body>
</html>
